<?php

namespace App\DataTables;

use App\Models\UserAttendance;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class UserAttendanceDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        return datatables()
            ->eloquent($query)
            // Customize columns here
            ->editColumn('customer_id', function ($row) {
                return $row->customer->CustomerName ?? ''; // Display customer name
            })
            ->editColumn('check_in_time', function ($row) {
                return $row->check_in_time ? $row->check_in_time->format('d-m-Y H:i:s') : ''; // Format check-in time
            })
            ->editColumn('check_out_time', function ($row) {
                return $row->check_out_time ? $row->check_out_time->format('d-m-Y H:i:s') : ''; // Format check-out time
            })
            ->editColumn('checkin_lat', function ($row) {
                return '<a href="https://www.google.com/maps?q=' . $row->checkin_lat . ',' . $row->checkin_lng . '" target="_blank">Show On Map</a>'; // Check-in location link
            })
            ->editColumn('checkout_lat', function ($row) {
                return $row->checkout_lat && $row->checkout_lng ? '<a href="https://www.google.com/maps?q=' . $row->checkout_lat . ',' . $row->checkout_lng . '" target="_blank">Show On Map</a>' : ''; // Check-out location link
            })
            ->editColumn('time_difference', function ($row) {
                if ($row->check_in_time && $row->check_out_time) {
                    $checkIn = $row->check_in_time;
                    $checkOut = $row->check_out_time;
                    $difference = $checkIn->diff($checkOut); // Calculate time difference
                    return $difference->format('%h hours %i minutes'); // Display time difference in hours and minutes
                }
                return ''; // If no check-out time, return empty
            })
            ->rawColumns(['checkin_lat', 'checkout_lat']); // Allow HTML in the columns
    }


    /**
     * Get query source of dataTable.
     *
     * @param \App\Models\UserAttendance $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(UserAttendance $model)
    {
        return $model->fetchAllUserAttendance($this->teammember_id,$this->month,$this->year); // Query the user_attendances table
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->setTableId('user_attendances_table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->dom('Bfrtip')
            ->orderBy(1)
            ->buttons(
                Button::make('create'),
                Button::make('export'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            );
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns()
    {
        return [
            Column::make('id'),
            Column::make('customer_id')->title('Team Member Name'),
            Column::make('check_in_time')->title('Check In Time'),
            Column::make('checkin_lat')->title('Location'),
            Column::make('check_out_time')->title('Check Out Time'),
            Column::make('checkout_lat')->title('Location'),
            Column::computed('time_difference')->title('Time Difference'),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    protected function filename()
    {
        return 'UserAttendance_' . date('YmdHis');
    }
}
