<?php

namespace App\Http\Controllers;

use App\DataTables\AreaDatatable;
use App\Models\Area;
use App\Exports\CommonExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CommonSessionImport;
use Carbon\Carbon;

class AreaController extends Controller
{
    //

    public function index(Request $request, AreaDatatable $dataTable)
    {

        return $dataTable->render('areas.index');
    }
    public function addEdit(Request $request, $id = null)
    {
        try {
            $type = empty($id) ? 'Add' : 'Edit';
            $area = Area::find($id);
            // $cities = DB::table('cities')->select('id', 'name')->get();

            $cities = DB::table('AddressDataTBL')->select('district as name', 'district as id')->groupBy('district')->orderBy('district', 'asc')->get();

            $html = view('areas.addEditModal', compact('area', 'type', 'cities'))->render();
            return response()->json([
                "status" => true,
                "html" => $html,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "message" => $e->getMessage(),
            ], 500);
        }
    }
    public function store(Request $request, $id = null)
    {
        try {

            $validator = Validator::make($request->all(), [
                'city_name' => 'required|string',
                'name' => 'required|string',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    "status" => false,
                    "error" => $validator->errors(),
                    "message" => $validator->errors()->first(),
                ], 422);
            }

            $exists = Area::where('city_name', $request->city_name)->where('name', $request->name)->where('id', '!=', $id)->exists();
            if ($exists) {
                return response()->json([
                    "status" => false,
                    "error" => ["name" => "Area already exists"],
                ], 422);
            }
            $area = Area::updateOrCreate(['id' => $id], $request->all());
            return response()->json([
                "status" => true,
                "message" => 'Area ' . ($id ? 'updated' : 'created') . ' successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "message" => $e->getMessage(),
            ], 500);
        }
    }
    public function destroy(Request $request, $id)
    {
        try {
            Area::find($id)->delete();
            return response()->json([
                "status" => true,
                "message" => 'Area deleted successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "message" => $e->getMessage(),
            ], 500);
        }
    }

    public function importOpen(Request $request)
    {
        try {
            $html = view('areas.importModal')->render();
            return response()->json([
                "status" => true,
                "html" => $html,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "message" => $e->getMessage(),
            ], 500);
        }
    }
    public function importArea(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'excel_file' => 'required|mimes:csv,xlsx,xls',
            ]);
            if ($validator->fails()) {
                return response()->json([
                    "status" => false,
                    "error" => $validator->errors(),
                    "message" => $validator->errors()->first(),
                ], 422);
            }
            $file = $request->file('excel_file');
            $key = 'area_import';
            session()->forget($key);
            Excel::import(new CommonSessionImport($key), $request->file('excel_file'));

            //validate data
            $rows = session($key);
            $rowCount = count($rows);

            //check any city is empty
            $totalCity = collect($rows)->pluck('city')->filter()->count();
            if ($rowCount != $totalCity) {
                return response()->json([
                    'message' => "Some city is empty",
                    "status" => false,
                ], 400);
            }

            //check any city is empty
            $totalArea = collect($rows)->pluck('area')->filter()->count();
            if ($rowCount != $totalArea) {
                return response()->json([
                    'message' => "Some area is empty",
                    "status" => false,
                ], 400);
            }

            $cityList = collect($rows)->pluck('city')->unique()->toArray();
            $totalCity = count($cityList);
            $cityList = DB::table('AddressDataTBL')->whereIn(DB::raw('LOWER(district)'), array_map('strtolower', $cityList))->groupBy('district')->get();


            if($totalCity != count($cityList)){
                return response()->json([
                    'message' => "Some city is not available in city master",
                    "status" => false,
                ], 400);
            }


            $data = [];
            foreach ($rows as $row) {
              $cityFromDB = DB::table('AddressDataTBL')->where(DB::raw('LOWER(district)'), strtolower($row['city']))->first();


                Area::updateOrCreate(['name' => $row['area'], 'city_name' => $cityFromDB?->district ?? $row['city']]);
            }
            return response()->json([
                "status" => true,
                "message" => 'Area imported successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status" => false,
                "message" => $e->getMessage(),
            ], 500);
        }
    }

    public function exportArea(Request $request)
    {
        $areas = Area::fetchAllData()->get()->map(function ($area) {
            return [
                'col1' => $area?->city_name,
                'col2' => $area?->name,
            ];
        });

        return Excel::download(new CommonExport($areas, ['City', 'Area']), 'areas.xlsx');
    }
}
